//
// CallCobol
//
// Copyright (C) 2010-2023 Heirloom Computing Inc.  All Rights Reserved.
//
// This file and associated files are copyrighted information
// of Heirloom Computing.  Permission is granted for usage in 
// conjunction with the Elastic COBOL product.
//
// This Java class demonstrates calling a COBOL program (PROGRAM-ID style)
// from Java.  A number of parameters are passed to the linkage section
// of the COBOL program by using the parameterList object.
//
// The COBOL program called is 'cobolprog'.

public class CallCobol
{
	static public void main(String[] args)
	{
		// 1. Create an Object array, populating it with parameters.
		//    The parameters may be of type String, descendents of
		//    java.lang.Number such as Integer, BigDecimal, and
		//    implementors of com.heirloomcomputing.ecs.api.Datatype.  Other types
		//    will be converted to String upon demand.
		
		Object[] params=new Object[]
		{
			new Integer(1),	// int
			"two",				// String
			new Double(3.0),	// double
			new Long(4),		// long
			new Character('5')	// char			
		};
		
		// 2. Create a boolean array, populating it with true
		//    for passing BY REFERENCE, false for passing
		//    BY CONTENT.  The array itself may be null.
		//    For Java calling COBOL, BY REFERENCE is actually
		//    passing by COPY/RESTORE.
		
		boolean[] convention=new boolean[]
		{
			false,
			false,
			false,
			true,
			false
		};
    
		// 2. Instantiate the COBOL program; the classname
		//    is the same as the program-id, converted to
		//    lowercase, with -'s as _'s.  (A COBOL program
		//    with a "literal" name is preserved identically.)

		//    Each instance created of the COBOL program is a
		//    separate entity with its own W-S, FILE section, etc.
		//    The _same_ instance may be called repeatedly to
		//    reuse the same W-S, FILE section, etc. just as a
		//    COBOL program normally will reuse the data when
		//    CALLed by another COBOL program.
	    
		cobolprog progToCall=new cobolprog(); // The program-id, all lowercase
    
		// 3. Call the COBOL using the call method.
		try 
		{
			System.out.println("Java program --> Calling COBOL");
		
			// This result is the the object referenced by the
			// PROCEDURE DIVISION USING ... GIVING|RETURNING clause
			//
			// If the GIVING refers to an Object, then the Object
			// is of that type.  If the GIVING refers to a traditional
			// COBOL variable, use the .toString() method to obtain
			// its contents.
		
			// params may be omitted if there are no parameters to pass.
		
			Object result=progToCall.call(convention,params);
	
			System.out.println("Java program --> Called COBOL and returned successfully!");
			System.out.println("COBOL program function result="+result);
	    } 
		catch (Exception e) 
		{
			System.out.println("An exception happened!");
			e.printStackTrace(System.out);
		}
		
		// Display parameters passed back by reference
		if(convention!=null)
		{
			for(int i=0;i<params.length && i<convention.length;i++)
			{
				if(convention[i])
					System.out.println("return parameter "+i+": '"+params[i]+"'");
			}
		}		
	} // end main
} // end CallCobol class

