/**
 * DDSJava
 *
 * Copyright (C) 2010-2023 Heirloom Computing Inc.  All Rights Reserved.
 *
 * Sample for Elastic COBOL DDS Plug-in for Java.
 */

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.util.Vector;

import javax.swing.DefaultCellEditor;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.border.CompoundBorder;
import javax.swing.border.EmptyBorder;
import javax.swing.border.EtchedBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

import com.heirloomcomputing.dds.DDSController;

public class DDSInteractive extends JFrame implements WindowListener, ActionListener, ItemListener
{
    static public final String DEFAULT_DDSFILE="DDSFILE";

    // We cannot call DDSController action methods from within an AWT/JFC thread, so we use a synchronized wait/notify
    static private final Object lockObject=new Object();

    // Declare the main dds controller class
    private DDSController ddsController=null;

    // Declare globally accessible graphics components
    private JComboBox recordNameCombo=null;
    private JComboBox actionCombo=null;
    private JLabel startingAtLabel=null;
    private JTextField startingAtField=null;
    private JLabel subfileRecordNumberLabel=null;
    private JTextField subfileRecordNumberField=null;
    private JTable fieldTable=null;
    private JCheckBox[] indicatorCheckBoxes=null;

    // The terminal window
    private JDialog displayWindow=null;

    // Used to determine whether or not the record names combo has been initialized
    private boolean initializedRecordNames=false;

    // Used to flag whether or not the display window was initialized
    private boolean windowInitialized=false;

    // Used to save off the amount of fields added to table last
    private int lastNumFields=0;

    // Used to call DDSController action methods without having to be in an AWT/JFC Thread
    private Thread actionThread=null;

    // Constructors
    public DDSInteractive() { this(DEFAULT_DDSFILE); }
    public DDSInteractive(String ddsFilename)
    {
        ddsController=new DDSController(ddsFilename);

        if (ddsController.open())
        {
            initialize();
        }
        else
        {
            JOptionPane.showMessageDialog(this, "Unable to open file '"+ddsController.getFilename()+"'","Error",JOptionPane.ERROR_MESSAGE);
            exit();
        }
    }

    private void initialize()
    {
        setTitle("Elastic COBOL DDS Plug-in Interactive Sample Program");
        initializeGraphics();
        initializeValues();
        pack();
        setVisible(true);
        addWindowListener(this);

        // We should never call DDController actions from within an AWT/JFC thread,
        // so we have a separate thread wait to be notified of when to perform the
        // action.
        try {
            actionThread=new Thread(new Runnable()
            {
                public void run()
                {
                    while (true)
                    {
                        synchronized (lockObject)
                        {
                            try { lockObject.wait(); } catch (Exception e) {;}
                            performAction();
                        }
                    }
                }
            }
            );
            actionThread.start();
        } catch (Exception e)
        {
            JOptionPane.showMessageDialog(this, "Problem initializing thread","Error",JOptionPane.ERROR_MESSAGE);
            exit();
        }
    }

    private void initializeGraphics()
    {
        // The screen looks like this:

        //
        //                                           Indicator values for current record:
        // Filename:              DDSFILE            (RED=Option, BLU=Response, GRN=Both)
        //                                               0  1  2  3  4  5  6  7  8  9
        // Record name:           _____________V       0
        //                                             1
        // Action:                _____________V       2
        //                                             3
        // Starting at line:      _____________        4
        //                                             5
        // Subfile record number: _____________        6
        //                                             7
        //                                             8
        //                                             9
        //
        // Field values for current record:
        // +--------------+----------+-------------------------------------+
        // | Field name   | datatype |  Value                              |
        // +--------------+----------+-------------------------------------+
        // |              |          |                                     |
        // +--------------+----------+-------------------------------------+
        // |              |          |                                     |
        // +--------------+----------+-------------------------------------+
        // |              |          |                                     |
        // +--------------+----------+-------------------------------------+
        // |              |          |                                     |
        // +--------------+----------+-------------------------------------+
        //
        //           +---------------+  +-------------------+ +------------+
        //           |  Clear Values |  |  Perform Action   | |   Exit     |
        //           +---------------+  +-------------------+ +------------+
        //

        JLabel filenameLabel=new JLabel("  Filename:");
        JLabel recordNameLabel=new JLabel("  Record name:");
        JLabel actionLabel=new JLabel("  Action:");
        startingAtLabel=new JLabel("  Starting at line:");
        subfileRecordNumberLabel=new JLabel("  Subfile record number:");

        JPanel panel1=new JPanel();
        panel1.setLayout(new GridLayout(6,1));
        panel1.add(filenameLabel);
        panel1.add(recordNameLabel);
        panel1.add(actionLabel);
        panel1.add(startingAtLabel);
        panel1.add(subfileRecordNumberLabel);
        panel1.add(new JLabel(""));

        JLabel filenameLabel2=new JLabel(ddsController.getFilename());
        recordNameCombo=new JComboBox();
        actionCombo=new JComboBox();
        startingAtField=new JTextField();
        subfileRecordNumberField=new JTextField();

        JPanel panel2=new JPanel();
        panel2.setLayout(new GridLayout(6,1));
        panel2.add(filenameLabel2);
        panel2.add(recordNameCombo);
        panel2.add(actionCombo);
        panel2.add(startingAtField);
        panel2.add(subfileRecordNumberField);
        panel2.add(new JLabel(""));

        JPanel actionPanel=new JPanel();
        actionPanel.setLayout(new BorderLayout(20,10));
        actionPanel.add("West", panel1);
        actionPanel.add("Center",panel2);

        JPanel panel3=new JPanel();
        panel3.setLayout(new BorderLayout());
        panel3.add("North", actionPanel);

        Dimension d3=panel3.getPreferredSize();
        panel3.setPreferredSize(new Dimension(300,d3.height));

        // The indicator table really isn't a table, but a JPanel with a 11 by 11 grid
        // (Column and row labels and a 10x10 grid of checkboxes representing the indicators)
        JPanel indicatorTable=new JPanel();
        indicatorTable.setLayout(new GridLayout(11,11));

        JLabel label=new JLabel(" ");
        label.setPreferredSize(new Dimension(10,15));
        indicatorTable.add(label); // The top left of the grid is blank

        for (int i=0; i<10; i++)
        {
            label=new JLabel(""+i);
            label.setPreferredSize(new Dimension(10,15));
            indicatorTable.add(label);
        }

        indicatorCheckBoxes=new JCheckBox[100]; // 1-99 has checkboxes, 0 does not...
        for (int i=0; i<10; i++)
        {
            label=new JLabel(""+i);
            label.setPreferredSize(new Dimension(10,15));
            indicatorTable.add(label);
            for (int j=0; j<10; j++)
            {
                JCheckBox checkBox=new JCheckBox();
                checkBox.setHorizontalAlignment(JCheckBox.CENTER);
                checkBox.setHorizontalTextPosition(JCheckBox.CENTER);
                checkBox.setMargin(new Insets(1,1,1,1));
                checkBox.setPreferredSize(new Dimension(10,15));

                // checkBox.setOpaque(true);
                indicatorTable.add(checkBox);
                if ((i==0) && (j==0))
                {
                    checkBox.setVisible(false);
                    continue;
                }
                indicatorCheckBoxes[(i*10)+j]=checkBox;
            }
        }

        indicatorTable.setPreferredSize(new Dimension((int)(10*11*1.7),15*11));
        indicatorTable.setBorder(new EmptyBorder(15,35,1,1)); // We need to try to center the grid

        String[][] tableData=new String[100][6]; // Up to 100 rows of field info
        for (int i=0; i<100; i++) { tableData[i][0]=""; tableData[i][1]=""; tableData[i][2]=""; tableData[i][3]=""; tableData[i][4]=""; tableData[i][5]=""; }

        String[] tableHeader=new String[] { "Field name", "Length", "Dec Pos", "Datatype", "Usage", "Value"};

        fieldTable=new JTable(tableData, tableHeader)
        {
            public boolean isCellEditable(int row, int column)
            {
                if (column!=5) return false;
                String val=(String)getValueAt(row,0);
                if ((val==null)||(val.trim().length()==0)) return false;

                val=(String)getValueAt(row,4);
                if ((val==null)||(val.length()==0)) return false;

                char usage=val.charAt(0);
                if (usage!='I') return true;

                return false;
            }
        };

        fieldTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);

        TableColumnModel model=fieldTable.getColumnModel();
        TableColumn column;

        column = model.getColumn(0);
        column.setPreferredWidth(120);
        column = model.getColumn(1);
        column.setPreferredWidth(60);
        column = model.getColumn(2);
        column.setPreferredWidth(60);
        column = model.getColumn(3);
        column.setPreferredWidth(70);
        column = model.getColumn(4);
        column.setPreferredWidth(50);
        column = model.getColumn(5);
        column.setPreferredWidth(220);

        JScrollPane scroller=new JScrollPane(fieldTable);
        scroller.setPreferredSize(new Dimension(580,160));

        JPanel labelPanel=new JPanel();
        labelPanel.setLayout(new BorderLayout());
        labelPanel.add("North",new JLabel("   Indicator values for current record:"));
        labelPanel.add("South",new JLabel("(RED=Option, BLU=Response, GRN=Both)"));

        JPanel panel4=new JPanel();
        panel4.setLayout(new BorderLayout());
        panel4.add("North",labelPanel);
        panel4.add("West", indicatorTable);

        JPanel panel5=new JPanel();
        panel5.setLayout(new BorderLayout(30,10));
        panel5.add("West", panel3);
        panel5.add("Center",panel4);

        JPanel panel6=new JPanel();
        panel6.setLayout(new BorderLayout());
        panel6.add("North", new JLabel("  Field values for current record:"));
        panel6.add("Center", scroller);

        JButton clearButton=new JButton("Clear values");
        JButton performButton=new JButton("Perform Action");
        JButton exitButton=new JButton("Exit");

        JPanel panel7=new JPanel();
        panel7.setLayout(new GridLayout(1,3,20,20));
        panel7.add(clearButton);
        panel7.add(performButton);
        panel7.add(exitButton);

        JPanel panel8=new JPanel();
        panel8.setLayout(new BorderLayout());
        panel8.add("East", panel7);

        Container contentPane=getContentPane();
        contentPane.setLayout(new BorderLayout(20,20));
        contentPane.add("North", panel5);
        contentPane.add("Center", panel6);
        contentPane.add("South", panel8);

        ((JComponent)contentPane).setBorder(new CompoundBorder(new EtchedBorder(EtchedBorder.LOWERED), new EmptyBorder(10,10,10,10)));

        clearButton.setActionCommand("CLEAR");
        performButton.setActionCommand("ACTION");
        exitButton.setActionCommand("EXIT");

        clearButton.addActionListener(this);
        performButton.addActionListener(this);
        exitButton.addActionListener(this);
    }

    private boolean initializeValues()
    {
        // Add values to the fields, combo-boxes and tables

        if (!initializedRecordNames)
        {
            initializedRecordNames=true;
            String[] recordNames=ddsController.getRecordNames();
            if (recordNames!=null)
            {
                int len=recordNames.length;
                for (int i=0; i<len; i++)
                {
                    recordNameCombo.addItem(recordNames[i]);
                }
                if (len>0) recordNameCombo.setSelectedItem(recordNames[0]);
            }
        }
        else
        {
            recordNameCombo.removeItemListener(this);
        }

        String currentRecord=(String)recordNameCombo.getSelectedItem();
        if (currentRecord==null) return false;

        ddsController.setCurrentRecord(currentRecord);

        boolean isSubfileRecord=ddsController.isSubfileRecord();

        if (!isSubfileRecord)
        {
            // Record is not a subfile record
            startingAtLabel.setVisible(true);
            startingAtField.setVisible(true);
            startingAtField.setEnabled(true);
            subfileRecordNumberField.setEnabled(false);
            subfileRecordNumberLabel.setVisible(false);
            subfileRecordNumberField.setVisible(false);
            startingAtField.setText("");
        }
        else
        {
            // Record is a subfile record
            startingAtField.setEnabled(false);
            startingAtLabel.setVisible(false);
            startingAtField.setVisible(false);
            subfileRecordNumberLabel.setVisible(true);
            subfileRecordNumberField.setVisible(true);
            subfileRecordNumberField.setEnabled(true);
            subfileRecordNumberField.setText("1");
        }

        if (actionCombo.getItemCount()>0) actionCombo.removeAllItems();

        if (!isSubfileRecord)
        {
            actionCombo.addItem("WRITE-READ");
        }
        else
        {
            actionCombo.addItem("WRITE-SUBFILE");
            actionCombo.addItem("READ-SUBFILE");
            actionCombo.addItem("READ-NEXT-MODIFIED");
            actionCombo.addItem("REWRITE-SUBFILE");
            actionCombo.setSelectedItem("WRITE-SUBFILE");
        }

        Vector optionIndicators=ddsController.getOptionIndicatorsAsVector();
        Vector responseIndicators=ddsController.getResponseIndicatorsAsVector();

        Color RED=new Color(255,150,150);
        Color BLUE=new Color(150,150,255);
        Color GREEN=new Color(150,255,150);

        // Option=red, response=blue, both=green, white="default unused indicator"
        for (int i=1; i<100; i++)
        {
            JCheckBox checkBox=indicatorCheckBoxes[i];
            checkBox.setSelected(false);

            boolean colorSet=false;
            if (optionIndicators!=null)
            {
                for (int j=0; j<optionIndicators.size(); j++)
                {
                    Integer curInt=(Integer)optionIndicators.elementAt(j);
                    if (curInt.intValue()==i)
                    {
                        checkBox.setBackground(RED);
                        colorSet=true;
                        break;
                    }
                }
            }

            if (responseIndicators!=null)
            {
                for (int j=0; j<responseIndicators.size(); j++)
                {
                    Integer curInt=(Integer)responseIndicators.elementAt(j);
                    if (curInt.intValue()==i)
                    {
                        if (colorSet)
                        {
                            checkBox.setBackground(GREEN);
                        }
                        else
                        {
                            checkBox.setBackground(BLUE);
                            colorSet=true;
                        }
                        break;
                    }
                }
            }

            if (!colorSet) checkBox.setBackground(Color.white);
        }

        if (fieldTable.isEditing()) fieldTable.editingCanceled(new ChangeEvent(this));
        fieldTable.clearSelection();

        int numFields=ddsController.getNumberOfFields();
        for (int i=0; i<numFields; i++)
        {
            String name=ddsController.getFieldName(i);
            if (name==null) continue;

            int len=ddsController.getFieldLength(i);
            int decpos=ddsController.getFieldDecimalPositions(i);
            char datatype=ddsController.getFieldDatatype(i);
            char usage=ddsController.getFieldUsage(i);
            Object value=ddsController.getFieldValue(i);
            if (value==null) value=new String("");

            fieldTable.setValueAt(name,i,0);
            fieldTable.setValueAt(""+len,i,1);
            fieldTable.setValueAt(""+decpos,i,2);
            fieldTable.setValueAt(""+datatype,i,3);
            fieldTable.setValueAt(""+usage,i,4);
            fieldTable.setValueAt(value.toString().trim(),i,5);
        }

        // Clear out the values from previous call if any
        for (int i=numFields; i<lastNumFields; i++)
        {
            fieldTable.setValueAt("",i,0);
            fieldTable.setValueAt("",i,1);
            fieldTable.setValueAt("",i,2);
            fieldTable.setValueAt("",i,3);
            fieldTable.setValueAt("",i,4);
            fieldTable.setValueAt("",i,5);
        }

        lastNumFields=numFields;

        recordNameCombo.addItemListener(this);

        return true;
    }

    private void performClear()
    {
        ddsController.clearAllIndicatorValues();
        ddsController.clearAllFieldValues();
        initializeValues();
    }

    private void prepareForWrite()
    {
        for (int i=1; i<100; i++)
        {
            JCheckBox checkBox=indicatorCheckBoxes[i];
            boolean value=checkBox.isSelected();
            ddsController.setIndicatorValue(i,value);
        }

        int numFields=ddsController.getNumberOfFields();
        for (int i=0; i<numFields; i++)
        {
            Object o=fieldTable.getValueAt(i,5);
            ddsController.setFieldValue(i,o);
        }
    }

    private void refreshAfterRead()
    {
        for (int i=1; i<100; i++)
        {
            JCheckBox checkBox=indicatorCheckBoxes[i];
            boolean value=ddsController.getIndicatorValue(i);
            checkBox.setSelected(value);
        }

        int numFields=ddsController.getNumberOfFields();
        for (int i=0; i<numFields; i++)
        {
            Object o=ddsController.getFieldValue(i);
            if (o==null) o=new String("");

            fieldTable.setValueAt(o.toString().trim(),i,5);
        }
    }

    private boolean initializeDisplayWindow()
    {
        Component displayComponent=ddsController.getDisplayComponent();
        if (displayComponent==null)
        {
            JOptionPane.showMessageDialog(this, "An error occurred initializing the display window", "Error", JOptionPane.ERROR_MESSAGE);
            return false;
        }

        displayWindow=new JDialog(this,ddsController.getFilename(),true);
        displayWindow.setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE);

        Container contentPane=displayWindow.getContentPane();
        contentPane.setLayout(new BorderLayout());
        contentPane.add("Center", displayComponent);

        displayWindow.pack();

        windowInitialized=true;
        return true;
    }

    private void performAction()
    {
        // Sometimes the user forgets to hit enter when editing a table value, so we force
        // the cell editor to stop editing...
        if (fieldTable.isEditing())
        {
            try { ((DefaultCellEditor)fieldTable.getCellEditor()).stopCellEditing(); } catch (Exception e) {;}
        }

        String action=(String)actionCombo.getSelectedItem();

        ddsController.clearAllIndicatorValues();
        ddsController.clearAllFieldValues();

        if ("WRITE-READ".equals(action))
        {
            if (!windowInitialized)
            {
                if (!initializeDisplayWindow()) return;
            }

            if (displayWindow!=null)
            {
                SwingUtilities.invokeLater(new Runnable() { public void run() { displayWindow.setVisible(true); } });
            }

            prepareForWrite();

            String text=startingAtField.getText();
            int startingAt;
            try { startingAt=Integer.valueOf(text).intValue(); } catch (Exception e) { startingAt=-1; }

            int result=ddsController.write(startingAt);

            if (result!=DDSController.ERROR_OK)
            {
                JOptionPane.showMessageDialog(this, "An error occurred during write", "Write error", JOptionPane.ERROR_MESSAGE);
                return;
            }

            result=ddsController.read();

            if ((windowInitialized) && (displayWindow!=null)) displayWindow.setVisible(false);

            if (result!=DDSController.ERROR_OK)
            {
                JOptionPane.showMessageDialog(this, "An error occurred during read", "Read error", JOptionPane.ERROR_MESSAGE);
                performClear();
                return;
            }

            refreshAfterRead();
        }
        else if ("WRITE-SUBFILE".equals(action))
        {
            prepareForWrite();

            String text=subfileRecordNumberField.getText();
            int subfileRecordNumber;
            try { subfileRecordNumber=Integer.valueOf(text).intValue(); } catch (Exception e) { subfileRecordNumber=1; }

            int result=ddsController.writeSubfile(subfileRecordNumber);
            if (result!=DDSController.ERROR_OK)
            {
                if (result==DDSController.ERROR_WRITE_SUBFILE_RECORD_ALREADY_EXISTS)
                {
                    JOptionPane.showMessageDialog(this, "Subfile record already exists", "Write error", JOptionPane.ERROR_MESSAGE);
                }
                else if (result==DDSController.WARNING_WRITE_SUBFILE_MAX_RECORDS_REACHED)
                {
                    JOptionPane.showMessageDialog(this, "Maximum subfile records reached", "Warning", JOptionPane.INFORMATION_MESSAGE);
                }
                else if (result==DDSController.WARNING_WRITE_SUBFILE_MAX_RECORDS_OVERFLOW)
                {
                    JOptionPane.showMessageDialog(this, "Maximum subfile records overflow", "Warning", JOptionPane.INFORMATION_MESSAGE);
                }
                else
                {
                    JOptionPane.showMessageDialog(this, "An error occurred during write", "Write error", JOptionPane.ERROR_MESSAGE);
                }
            }
            else
            {
                JOptionPane.showMessageDialog(this, "Subfile record written", "Notification", JOptionPane.INFORMATION_MESSAGE);
            }
        }
        else if ("READ-SUBFILE".equals(action))
        {
            String text=subfileRecordNumberField.getText();
            int subfileRecordNumber;
            try { subfileRecordNumber=Integer.valueOf(text).intValue(); } catch (Exception e) { subfileRecordNumber=1; }

            int result=ddsController.readSubfile(subfileRecordNumber);

            if (result!=DDSController.ERROR_OK)
            {
                if (result==DDSController.ERROR_READ_SUBFILE_RECORD_DOES_NOT_EXIST)
                {
                    JOptionPane.showMessageDialog(this, "Subfile record does not exist", "Read error", JOptionPane.ERROR_MESSAGE);
                }
                else
                {
                    JOptionPane.showMessageDialog(this, "An error occurred during read", "Read error", JOptionPane.ERROR_MESSAGE);
                }
                performClear();
                return;
            }

            JOptionPane.showMessageDialog(this, "Subfile record read successful", "Notification", JOptionPane.INFORMATION_MESSAGE);
            refreshAfterRead();
        }
        else if ("READ-NEXT-MODIFIED".equals(action))
        {
            int result=ddsController.readNextModifiedSubfile();
            if (result!=DDSController.ERROR_OK)
            {
                if (result==DDSController.ERROR_READ_NEXT_SUBFILE_RECORD_DOES_NOT_EXIST)
                {
                    JOptionPane.showMessageDialog(this, "Subfile record does not exist", "Read-next error", JOptionPane.ERROR_MESSAGE);
                }
                else
                {
                    JOptionPane.showMessageDialog(this, "An error occurred during read", "Read-next error", JOptionPane.ERROR_MESSAGE);
                }
                performClear();
                return;
            }

            JOptionPane.showMessageDialog(this, "Subfile record read successful", "Notification", JOptionPane.INFORMATION_MESSAGE);
            refreshAfterRead();
        }
        else if ("REWRITE-SUBFILE".equals(action))
        {
            prepareForWrite();

            String text=subfileRecordNumberField.getText();
            int subfileRecordNumber;
            try { subfileRecordNumber=Integer.valueOf(text).intValue(); } catch (Exception e) { subfileRecordNumber=1; }

            int result=ddsController.rewriteSubfile(subfileRecordNumber);
            if (result!=DDSController.ERROR_OK)
            {
                if (result==DDSController.ERROR_REWRITE_SUBFILE_RECORD_DOES_NOT_EXIST)
                {
                    JOptionPane.showMessageDialog(this, "Subfile record does not exist", "Rewrite error", JOptionPane.ERROR_MESSAGE);
                }
                else
                {
                    JOptionPane.showMessageDialog(this, "An error occurred during rewrite", "Rewrite error", JOptionPane.ERROR_MESSAGE);
                }
            }
            else
            {
                JOptionPane.showMessageDialog(this, "Subfile record rewritten", "Notification", JOptionPane.INFORMATION_MESSAGE);
            }
        }
    }

    private void exit()
    {
        try { if (actionThread!=null) actionThread.stop(); } catch (Exception e) {;}
        if (displayWindow!=null) displayWindow.dispose();
        dispose();
        if (ddsController!=null) ddsController.close();
        System.exit(0);
    }

    // Item listener method
    public void itemStateChanged(ItemEvent e)
    {
        if ((e!=null)&&(ddsController!=null)&&(e.getSource()==recordNameCombo)&&(e.getStateChange()==ItemEvent.SELECTED))
            initializeValues();
    }
    // End item listener method

    // Action listener method
    public void actionPerformed(ActionEvent e)
    {
        if (e==null) return;

        String actionCommand=e.getActionCommand();
        if (actionCommand==null) return;

        if ("EXIT".equals(actionCommand))
        {
            exit();
        }
        else if ("CLEAR".equals(actionCommand))
        {
            if (ddsController!=null) performClear();
        }
        else if ("ACTION".equals(actionCommand))
        {
            // We should never call DDController actions from within an AWT/JFC thread,
            // so we have a separate thread running in the background waiting to be
            // notified of when to perform the action.
            if (ddsController!=null)
            {
                synchronized (lockObject)
                {
                    try { lockObject.notifyAll(); } catch (Exception e2) {;}
                }
            }
        }
    }
    // End action listener method

    // Window listener methods
    public void windowClosing(WindowEvent e)
    {
        if ((e!=null)&&(e.getWindow()==this)) { exit();  }
    }
    public void windowActivated(WindowEvent e) {;}
    public void windowClosed(WindowEvent e) {;}
    public void windowDeactivated(WindowEvent e) {;}
    public void windowDeiconified(WindowEvent e) {;}
    public void windowIconified(WindowEvent e) {;}
    public void windowOpened(WindowEvent e) {;}
    // End window listener methods

    static public void main(String[] args)
    {
        DDSInteractive ddsInteractive;

        if ((args==null)||(args.length==0))
        {
            // Calling DDSInteractive with no parameters will attempt to load a DDS file
            // called DDSFILE.  This DDS file must be accessible from the classpath.
            ddsInteractive=new DDSInteractive();
        }
        else
        {
            // Calling DDSInteractive with one parameter will attempt to load the DDS file
            // given by the parameter.  This DDS file must be accessible from the classpath.
            ddsInteractive=new DDSInteractive(args[0]);
        }
    }
}
