/**
 * DDSJava
 *
 * Copyright (C) 2010-2023 Heirloom Computing Inc.  All Rights Reserved.
 *
 * Sample for Elastic COBOL DDS Plug-in for Java.
 */

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.Frame;
import java.awt.Panel;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import javax.swing.JFrame;

import com.heirloomcomputing.dds.DDSController;

class DDSJava
{
    static public final String TEST_FILE="DDSFILE";

    // A constant of 80 spaces used to clear alphanumeric field values
    static private final String SPACES="                                                                                ";

    private DDSController ddsController=null;
    private Frame displayFrame=null;

    public DDSJava()
    {
        ;
    }

    private void error(String text)
    {
        System.out.println("ERROR: "+text);
    }

    private boolean setupDisplay()
    {
        Component c=ddsController.getDisplayComponent();
        if (c==null) return false;

        boolean isJFC=!(c instanceof Panel);

        if (!isJFC)
        {
            displayFrame=new Frame();
            displayFrame.setLayout(new BorderLayout());
            displayFrame.add("Center", c);
        }
        else
        {
            displayFrame=new JFrame();
            Container contentPane=((JFrame)displayFrame).getContentPane();
            contentPane.setLayout(new BorderLayout());
            contentPane.add("Center",c);
        }

        displayFrame.setTitle("DDS Plug-in Java Sample");

        displayFrame.pack();
        displayFrame.setVisible(true);

        displayFrame.addWindowListener(new WindowAdapter()
        {
            public void windowClosing(WindowEvent evt)
            {
                exit();
            }
        });

        return true;
    }

    private void exit()
    {
        if (displayFrame!=null) displayFrame.dispose();
        if (ddsController!=null) ddsController.close();
        System.exit(0);
    }

    private int pause()
    {
        ddsController.setCurrentRecord("PAUSEREC");

        int status=ddsController.write(22);
        if (status!=DDSController.ERROR_OK) return status;

        return ddsController.read();
    }

    private int writeThenPause(int startingAtLine)
    {
        int status=ddsController.write(startingAtLine);
        if (status!=DDSController.ERROR_OK) return status;
        return pause();
    }

    private int writeRead(int startingAtLine)
    {
        int status=ddsController.write(startingAtLine);
        if (status!=DDSController.ERROR_OK) return status;
        status=ddsController.read();
        return status;
    }

    private int performRecordPositioningTests()
    {
        int status=DDSController.ERROR_OK;

        ddsController.clearAll();

        ddsController.setCurrentRecord("CLEARREC");
        status=ddsController.write(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("POSREC1");
        status=ddsController.write(1);
        if (status!=DDSController.ERROR_OK) return status;
        status=ddsController.write(5);
        if (status!=DDSController.ERROR_OK) return status;
        status=ddsController.write(9);
        if (status!=DDSController.ERROR_OK) return status;
        status=ddsController.write(13);
        if (status!=DDSController.ERROR_OK) return status;
        status=writeThenPause(17);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("POSREC2");
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("POSREC3");
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("POSREC4");
        status=writeThenPause(14);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("POSREC5");
        status=writeThenPause(2);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("POSREC6");
        status=writeThenPause(19);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("POSREC7");
        status=ddsController.write(5);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("POSREC8");
        status=writeThenPause(12);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("POSREC9");
        status=writeThenPause(16);

        return status;
    }

    private int performDisplayAttributeTests()
    {
        int status=DDSController.ERROR_OK;

        ddsController.clearAll();

        ddsController.setCurrentRecord("DISPREC1");
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("DISPREC2");
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("CLEARREC");
        status=ddsController.write(-1);
        if (status!=DDSController.ERROR_OK) return status;

        int dspatrval=32;

        ddsController.setCurrentRecord("DISPREC3");
        for (int loopIndex=5; loopIndex<21; loopIndex++)
        {
            ddsController.setFieldValue("PFIELD1", "" + (char)(dspatrval++) );
            status=ddsController.write(loopIndex);
            if (status!=DDSController.ERROR_OK) return status;
        }

        status=pause();
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("DISPREC3");
        for (int loopIndex=5; loopIndex<21; loopIndex++)
        {
            ddsController.setFieldValue("PFIELD1", "" + (char)(dspatrval++) );
            status=ddsController.write(loopIndex);
            if (status!=DDSController.ERROR_OK) return status;
        }
        status=pause();

        return status;
    }

    private int performOptionIndicatorTests()
    {
        int status=DDSController.ERROR_OK;

        ddsController.clearAll();

        ddsController.setCurrentRecord("OPINREC1");
        ddsController.setIndicatorValue(1,true);
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("OPINREC1");
        ddsController.setIndicatorValue(99,true);
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("OPINREC1");
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("OPINREC1");
        ddsController.setIndicatorValue(1,true);
        ddsController.setIndicatorValue(99,true);
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("OPINREC2");
        ddsController.setIndicatorValue(1,true);
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("OPINREC2");
        ddsController.setIndicatorValue(2,true);
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("OPINREC2");
        ddsController.setIndicatorValue(3,true);
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("OPINREC2");
        ddsController.setIndicatorValue(1,true);
        ddsController.setIndicatorValue(2,true);
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("OPINREC2");
        ddsController.setIndicatorValue(1,true);
        ddsController.setIndicatorValue(3,true);
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("OPINREC2");
        ddsController.setIndicatorValue(2,true);
        ddsController.setIndicatorValue(3,true);
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("OPINREC2");
        ddsController.setIndicatorValue(1,true);
        ddsController.setIndicatorValue(2,true);
        ddsController.setIndicatorValue(3,true);
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("OPINREC2");
        ddsController.setIndicatorValue(72,true);
        ddsController.setIndicatorValue(73,true);
        ddsController.setIndicatorValue(63,true);
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("OPINREC2");
        ddsController.setIndicatorValue(60,true);
        ddsController.setIndicatorValue(61,true);
        ddsController.setIndicatorValue(62,true);
        ddsController.setIndicatorValue(63,true);
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("OPINREC2");
        ddsController.setIndicatorValue(60,true);
        ddsController.setIndicatorValue(61,true);
        ddsController.setIndicatorValue(62,true);
        ddsController.setIndicatorValue(72,true);
        ddsController.setIndicatorValue(73,true);
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("OPINREC2");
        ddsController.setIndicatorValue(60,true);
        ddsController.setIndicatorValue(61,true);
        ddsController.setIndicatorValue(62,true);
        ddsController.setIndicatorValue(72,true);
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("OPINREC2");
        ddsController.setIndicatorValue(60,true);
        ddsController.setIndicatorValue(61,true);
        ddsController.setIndicatorValue(62,true);
        ddsController.setIndicatorValue(72,true);
        ddsController.setIndicatorValue(63,true);
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("OPINREC2");
        ddsController.setIndicatorValue(63,true);
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("OPINREC2");
        ddsController.setIndicatorValue(72,true);
        ddsController.setIndicatorValue(73,true);
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("OPINREC3");
        ddsController.setIndicatorValue(1,true);
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("OPINREC3");
        ddsController.setIndicatorValue(2,true);
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("OPINREC3");
        ddsController.setIndicatorValue(3,true);
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("OPINREC3");
        ddsController.setIndicatorValue(1,true);
        ddsController.setIndicatorValue(2,true);
        ddsController.setIndicatorValue(3,true);
        status=writeThenPause(-1);

        return status;
    }

    private int performMessageTests()
    {
        int status=DDSController.ERROR_OK;

        ddsController.clearAll();

        ddsController.setCurrentRecord("MSGREC1");
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("MSGREC2");
        ddsController.setFieldValue("MSGNBR","0021");
        ddsController.setFieldValue("MSGFILENM","QCPFMSG");
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("MSGREC2");
        ddsController.setIndicatorValue(1,true);
        ddsController.setFieldValue("MSGNBR","0021");
        ddsController.setFieldValue("MSGFILENM","QCPFMSG");
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("MSGREC2");
        ddsController.setIndicatorValue(1,true);
        ddsController.setIndicatorValue(25,true);
        ddsController.setFieldValue("MSGIDNUM","KBD0014");
        ddsController.setFieldValue("MSGNBR","0007");
        ddsController.setFieldValue("MSGFILENM","QCPFMSG");
        status=writeThenPause(-1);

        return status;
    }

    private int performEntryFieldAttributeTests()
    {
        int status=DDSController.ERROR_OK;

        ddsController.clearAll();

        ddsController.setCurrentRecord("EFAREC1");
        status=writeRead(-1);

        return status;
    }

    private int performInputAttributeTests()
    {
        int status=DDSController.ERROR_OK;

        ddsController.clearAll();

        ddsController.setCurrentRecord("INPREC1");
        status=writeRead(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("INPREC2");
        ddsController.setFieldValue("PFIELD1",""+(char)48);
        status=writeRead(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.clearAll();

        ddsController.setFieldValue("PFIELD1",""+(char)48);
        ddsController.setIndicatorValue(1,true);
        status=writeRead(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("INPREC3");
        status=writeRead(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.clearAll();

        ddsController.setIndicatorValue(1,true);
        status=writeRead(-1);

        return status;
    }

    private int performCursorPositioningTests()
    {
        int status=DDSController.ERROR_OK;

        ddsController.clearAll();

        ddsController.setCurrentRecord("CPREC1");
        ddsController.setFieldValue("FIELD1","CURSOR HERE, POS=2");
        ddsController.setFieldValue("FIELD2",SPACES);
        ddsController.setFieldValue("FIELD3",SPACES);
        ddsController.setFieldValue("LINNBR","8");
        ddsController.setFieldValue("POSNBR","3");
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("CPREC1");
        ddsController.setFieldValue("FIELD1",SPACES);
        ddsController.setFieldValue("FIELD2","HERE");
        ddsController.setFieldValue("FIELD3",SPACES);
        ddsController.setFieldValue("LINNBR","8");
        ddsController.setFieldValue("POSNBR","60");
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("CPREC1");
        ddsController.setFieldValue("FIELD1",SPACES);
        ddsController.setFieldValue("FIELD2",SPACES);
        ddsController.setFieldValue("FIELD3","CURSOR HERE, POS=4");
        ddsController.setFieldValue("LINNBR","10");
        ddsController.setFieldValue("POSNBR","5");
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("CPREC1");
        ddsController.setFieldValue("FIELD1",SPACES);
        ddsController.setFieldValue("FIELD2","HERE");
        ddsController.setFieldValue("FIELD3",SPACES);
        ddsController.setFieldValue("LINNBR","8");
        ddsController.setFieldValue("POSNBR","60");
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("CPREC1");
        ddsController.setFieldValue("FIELD1","CURSOR HERE, POS=10");
        ddsController.setFieldValue("FIELD2",SPACES);
        ddsController.setFieldValue("FIELD3",SPACES);
        ddsController.setFieldValue("LINNBR","8");
        ddsController.setFieldValue("POSNBR","11");
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("CPREC2");
        ddsController.setFieldValue("FIELD1","CURSOR HERE");
        ddsController.setFieldValue("FIELD2",SPACES);
        ddsController.setFieldValue("FIELD3",SPACES);
        ddsController.setIndicatorValue(1,true);
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("CPREC2");
        ddsController.setFieldValue("FIELD1",SPACES);
        ddsController.setFieldValue("FIELD2","HERE");
        ddsController.setFieldValue("FIELD3",SPACES);
        ddsController.setIndicatorValue(2,true);
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("CPREC2");
        ddsController.setFieldValue("FIELD1",SPACES);
        ddsController.setFieldValue("FIELD2",SPACES);
        ddsController.setFieldValue("FIELD3","CURSOR HERE");
        ddsController.setIndicatorValue(3,true);
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("CPREC2");
        ddsController.setFieldValue("FIELD1",SPACES);
        ddsController.setFieldValue("FIELD2","HERE");
        ddsController.setFieldValue("FIELD3",SPACES);
        ddsController.setIndicatorValue(2,true);
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("CPREC2");
        ddsController.setFieldValue("FIELD1","CURSOR HERE");
        ddsController.setFieldValue("FIELD2",SPACES);
        ddsController.setFieldValue("FIELD3",SPACES);
        status=writeThenPause(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("CPREC3");
        status=writeRead(-1);

        return status;
    }

    private int performKeyboardControlTests()
    {
        int status=DDSController.ERROR_OK;

        ddsController.clearAll();

        ddsController.setCurrentRecord("KBDCREC1");
        ddsController.setFieldValue("FIELD1","-12.34");
        ddsController.setFieldValue("FIELD2","-12.34");
        ddsController.setFieldValue("FIELD3","-12.34");
        ddsController.setFieldValue("FIELD4","-12.34");
        ddsController.setFieldValue("FIELD5","ABC");
        ddsController.setFieldValue("FIELD6","ABC");

        String result=null;

        while(true)
        {
            ddsController.setFieldValue("RFIELD",SPACES);
            status=writeRead(-1);
            if (status!=DDSController.ERROR_OK) return status;

            result=ddsController.getFieldValueAsString("RFIELD");
            if (result==null) continue;

            result=result.trim();
            if ("Q".equalsIgnoreCase(result)) break;
        }

        return status;
    }

    private int performDataValidationTests()
    {
        int status=DDSController.ERROR_OK;

        while (true)
        {
            ddsController.clearAll();

            ddsController.setCurrentRecord("DVALREC1");
            status=writeRead(-1);
            if (status!=DDSController.ERROR_OK) return status;

            String result=ddsController.getFieldValueAsString("RFIELD");
            if (result==null) continue;

            result=result.trim();
            if ("Q".equalsIgnoreCase(result)) break;
        }

        while (true)
        {
            ddsController.clearAll();

            ddsController.setCurrentRecord("DVALREC2");
            status=writeRead(-1);
            if (status!=DDSController.ERROR_OK) return status;

            String result=ddsController.getFieldValueAsString("RFIELD");
            if (result==null) continue;

            result=result.trim();
            if ("Q".equalsIgnoreCase(result)) break;
        }

        while (true)
        {
            ddsController.clearAll();

            ddsController.setCurrentRecord("DVALREC3");
            status=writeRead(-1);
            if (status!=DDSController.ERROR_OK) return status;

            String result=ddsController.getFieldValueAsString("RFIELD");
            if (result==null) continue;

            result=result.trim();
            if ("Q".equalsIgnoreCase(result)) break;
        }

        while (true)
        {
            ddsController.clearAll();

            ddsController.setCurrentRecord("DVALREC4");
            status=writeRead(-1);
            if (status!=DDSController.ERROR_OK) return status;

            String result=ddsController.getFieldValueAsString("RFIELD");
            if (result==null) continue;

            result=result.trim();
            if ("Q".equalsIgnoreCase(result)) break;
        }

        while (true)
        {
            ddsController.clearAll();

            ddsController.setCurrentRecord("DVALREC5");
            status=writeRead(-1);
            if (status!=DDSController.ERROR_OK) return status;

            String result=ddsController.getFieldValueAsString("RFIELD");
            if (result==null) continue;

            result=result.trim();
            if ("Q".equalsIgnoreCase(result)) break;
        }

        return status;
    }

    // Called from performEditKeywordTests()
    private int performEditKeywordTestLoop()
    {
        String result;
        int status;

        while (true)
        {
            ddsController.setFieldValue("RFIELD",SPACES);
            status=writeRead(-1);
            if (status!=DDSController.ERROR_OK) return status;

            result=ddsController.getFieldValueAsString("RFIELD");
            if (result==null) continue;

            result=result.trim();
            if ("Q".equalsIgnoreCase(result)) break;
        }

        return status;
    }

    private int performEditKeywordTests()
    {
        int status=DDSController.ERROR_OK;

        ddsController.clearAll();

        ddsController.setCurrentRecord("EDITREC1");
        status=performEditKeywordTestLoop();
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("EDITREC2");
        status=performEditKeywordTestLoop();
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("EDITREC3");
        status=performEditKeywordTestLoop();
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("EDITREC4");
        status=performEditKeywordTestLoop();
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("EDITREC5");
        status=performEditKeywordTestLoop();

        return status;
    }

    private int performResponseIndicatorTests()
    {
        int status=DDSController.ERROR_OK;

        while (true)
        {
            ddsController.clearAll();
            ddsController.setCurrentRecord("REINREC1");
            status=writeRead(-1);
            if (status!=DDSController.ERROR_OK) return status;
            if (ddsController.getIndicatorValue(4)) break;
        }

        while (true)
        {
            ddsController.clearAll();
            ddsController.setCurrentRecord("REINREC2");
            status=ddsController.write(-1);
            if (status!=DDSController.ERROR_OK) return status;

            ddsController.setFieldValue("FIELD1","BYE");
            ddsController.setFieldValue("FIELD2","HELLO");

            status=ddsController.read();
            if (status!=DDSController.ERROR_OK) return status;

            if (ddsController.getIndicatorValue(1))
            {
                Object val=ddsController.getFieldValue("FIELD1");
                ddsController.setFieldValue("FIELD3",val);

                val=ddsController.getFieldValue("FIELD2");
                ddsController.setFieldValue("FIELD4",val);

                status=writeRead(-1);
                if (status!=DDSController.ERROR_OK) return status;

                continue;
            }

            break;
        }

        ddsController.setCurrentRecord("REINREC3");

        while (true)
        {
            ddsController.clearAll();
            status=writeRead(-1);
            if (status!=DDSController.ERROR_OK) return status;

            if (!ddsController.getIndicatorValue(1)) continue;

            break;
        }

        ddsController.setCurrentRecord("REINREC4");
        ddsController.setIndicatorValue(2,true);

        while (true)
        {
            status=writeRead(-1);
            if (status!=DDSController.ERROR_OK) return status;

            boolean ind01=ddsController.getIndicatorValue(1);
            boolean ind02=ddsController.getIndicatorValue(2);
            boolean ind03=ddsController.getIndicatorValue(3);
            boolean ind04=ddsController.getIndicatorValue(4);
            boolean ind05=ddsController.getIndicatorValue(5);

            if (ind01 || ind02 || ind03 || ind04 || ind05)
            {
                Object val=ddsController.getFieldValue("INFIELD");
                ddsController.setFieldValue("FIELD3",val);

                String text="";

                text+=(ind01 ? "1" : "0");
                text+=(ind02 ? "1" : "0");
                text+=(ind03 ? "1" : "0");
                text+=(ind04 ? "1" : "0");
                text+=(ind05 ? "1" : "0");

                ddsController.setFieldValue("FIELD1",text);
                ddsController.setFieldValue("FIELD2",text);

                continue;
            }
            break;
        }

        return status;
    }

    private int performMiscOverlayTests()
    {
        int status=DDSController.ERROR_OK;

        ddsController.clearAll();

        ddsController.setCurrentRecord("MSCOREC1");
        status=ddsController.write(-1);
        if (status!=DDSController.ERROR_OK) return status;

        try { Thread.sleep(4000); } catch (Exception e) {;}

        ddsController.setCurrentRecord("MSCOREC2");
        ddsController.setFieldValue("FIELD1","123");
        ddsController.setFieldValue("FIELD2","FIELD 2");
        status=ddsController.write(-1);
        if (status!=DDSController.ERROR_OK) return status;

        try { Thread.sleep(4000); } catch (Exception e) {;}

        ddsController.setCurrentRecord("MSCOREC3");
        ddsController.setFieldValue("FIELD1","FIELD 3");
        ddsController.setFieldValue("FIELD2","FIELD 4");
        status=ddsController.write(-1);
        if (status!=DDSController.ERROR_OK) return status;

        try { Thread.sleep(4000); } catch (Exception e) {;}

        ddsController.setCurrentRecord("MSCOREC4");
        ddsController.setFieldValue("FIELD1","FIELD 5");
        ddsController.setFieldValue("FIELD2","FIELD 6");
        status=ddsController.write(-1);
        if (status!=DDSController.ERROR_OK) return status;

        try { Thread.sleep(4000); } catch (Exception e) {;}

        status=ddsController.read();
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("MSCOREC5");
        status=writeRead(-1); // We don't support INZRCD keyword at low level

        return status;
    }

    private int performDateTimeTests()
    {
        int status=DDSController.ERROR_OK;

        String result;

        ddsController.clearAll();

        ddsController.setCurrentRecord("DATEREC1");

        while (true)
        {
            ddsController.setFieldValue("RFIELD",SPACES);
            status=writeRead(-1);
            if (status!=DDSController.ERROR_OK) return status;

            result=ddsController.getFieldValueAsString("RFIELD");
            if (result==null) continue;

            result=result.trim();
            if ("Q".equalsIgnoreCase(result)) break;
        }

        ddsController.setCurrentRecord("DATEREC2");

        while (true)
        {
            ddsController.setFieldValue("RFIELD",SPACES);
            status=writeRead(-1);
            if (status!=DDSController.ERROR_OK) return status;

            result=ddsController.getFieldValueAsString("RFIELD");
            if (result==null) continue;

            result=result.trim();
            if ("Q".equalsIgnoreCase(result)) break;
        }

        ddsController.setCurrentRecord("DATEREC3");

        while (true)
        {
            ddsController.setFieldValue("RFIELD",SPACES);
            status=writeRead(-1);
            if (status!=DDSController.ERROR_OK) return status;

            result=ddsController.getFieldValueAsString("RFIELD");
            if (result==null) continue;

            result=result.trim();
            if ("Q".equalsIgnoreCase(result)) break;
        }

        return status;
    }

    private int performErrorMessageTests()
    {
        int status=DDSController.ERROR_OK;

        ddsController.clearAll();

        ddsController.setCurrentRecord("EMSGREC1");
        status=writeRead(-1);
        if (status!=DDSController.ERROR_OK) return status;

        while (true)
        {
            ddsController.setIndicatorValue(1,true);
            status=writeRead(-1);
            if (status!=DDSController.ERROR_OK) return status;

            if (!ddsController.getIndicatorValue(1)) break;
        }

        while (true)
        {
            ddsController.setIndicatorValue(2,true);
            status=writeRead(-1);
            if (status!=DDSController.ERROR_OK) return status;

            if (!ddsController.getIndicatorValue(2)) break;
        }

        while (true)
        {
            ddsController.setIndicatorValue(3,true);
            ddsController.setFieldValue("RPLTXT","MY REPLACEMENT TEXT");
            status=writeRead(-1);
            if (status!=DDSController.ERROR_OK) return status;

            if (!ddsController.getIndicatorValue(3)) break;
        }

        status=writeRead(-1);

        return status;
    }

    private int performHelpRecordTests()
    {
        int status=DDSController.ERROR_OK;

        ddsController.clearAll();

        ddsController.setCurrentRecord("HELPREC1");
        status=ddsController.write(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("HELPREC2");
        status=ddsController.write(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("HELPREC3");
        status=ddsController.write(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("HELPREC1");
        status=ddsController.read();
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("HELPREC2");
        status=ddsController.read();
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("HELPREC3");
        status=ddsController.read();
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("HELPREC1");
        ddsController.setIndicatorValue(1,true);
        status=ddsController.write(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("HELPREC2");
        ddsController.setIndicatorValue(1,true);
        status=ddsController.write(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("HELPREC3");
        ddsController.setIndicatorValue(1,true);
        status=ddsController.write(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("HELPREC1");
        status=ddsController.read();
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("HELPREC2");
        status=ddsController.read();
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("HELPREC3");
        status=ddsController.read();

        return status;
    }

    private int performSubfileRecordTests()
    {
        int status=DDSController.ERROR_OK;

        ddsController.clearAll();

        ddsController.setCurrentRecord("SFLREC1");

        for (int i=1; i<5; i++)
        {
            status=ddsController.writeSubfile(i);
            if (DDSController.isError(status)) return status;
        }

        ddsController.setCurrentRecord("SFLCREC1");
        status=writeRead(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("SFLREC2");

        for (int i=1; i<16; i++)
        {
            String recNum=""+i;

            ddsController.setFieldValue("FIELD1",recNum);
            ddsController.setFieldValue("FIELD2",recNum);

            status=ddsController.writeSubfile(i);
            if (DDSController.isError(status)) return status;
        }

        ddsController.setCurrentRecord("SFLCREC2");
        status=writeRead(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("SFLREC3");

        for (int i=1; i<16; i++)
        {
            String recNum=""+i;

            ddsController.setFieldValue("FIELD1",recNum);
            ddsController.setFieldValue("FIELD2",recNum);

            status=ddsController.writeSubfile(i);
            if (DDSController.isError(status)) return status;
        }

        ddsController.setCurrentRecord("SFLCREC3");
        status=writeRead(-1);
        if (status!=DDSController.ERROR_OK) return status;

        //    ******************************************************
        //    * UNCOMMENT THE SECTION BELOW IF YOU WANT TO DISPLAY *
        //    * SUBFILE VALUES ON SYSOUT                           *
        //    ******************************************************

        /*
        ddsController.setCurrentRecord("SFLREC3");

        System.out.println("CALLING READ NEXT MODIFIED SUBFILE");
        while (!ddsController.isError(status))
        {
            status=ddsController.readNextModifiedSubfile();
            System.out.println("RECORD "+ddsController.getCurrentSubfileRecordNumber());
            System.out.println("FIELD1 = "+ddsController.getFieldValue("FIELD1"));
            System.out.println("FIELD2 = "+ddsController.getFieldValue("FIELD2"));
            System.out.println("STATUS = "+ddsController.getStatusDescription(status));
        }

        System.out.println("CALLING READ SUBFILE");
        for (int i=1; i<17; i++)
        {
            status=ddsController.readSubfile(i);
            System.out.println("RECORD "+ddsController.getCurrentSubfileRecordNumber());
            System.out.println("FIELD1 = "+ddsController.getFieldValue("FIELD1"));
            System.out.println("FIELD2 = "+ddsController.getFieldValue("FIELD2"));
            System.out.println("STATUS = "+ddsController.getStatusDescription(status));
        }
        System.out.println("DONE");
        */

        ddsController.setCurrentRecord("SFLREC4");

        for (int i=1; i<16; i++)
        {
            String recNum=""+i;

            ddsController.setFieldValue("FIELD1",recNum);
            ddsController.setFieldValue("FIELD2",recNum);

            status=ddsController.writeSubfile(i);
            if (DDSController.isError(status)) return status;
        }

        ddsController.setCurrentRecord("SFLCREC4");
        ddsController.setFieldValue("FIELD1","15");
        status=writeRead(-1);
        if (status!=DDSController.ERROR_OK) return status;

        ddsController.setCurrentRecord("SFLREC5");

        for (int i=1; i<16; i++)
        {
            String recNum=""+i;

            ddsController.setFieldValue("FIELD1",recNum);
            ddsController.setFieldValue("FIELD2",recNum);

            status=ddsController.writeSubfile(i);
            if (DDSController.isError(status)) return status;
        }

        ddsController.setCurrentRecord("SFLCREC5");
        ddsController.setFieldValue("FIELD1","6");
        status=writeRead(-1);
        if (status!=DDSController.ERROR_OK) return status;

        return status;
    }

    public boolean performTests()
    {
        ddsController=new DDSController(TEST_FILE);
        if (!ddsController.open())
        {
            error("Could not open file '"+TEST_FILE+"', make sure it is in the classpath.");
            return false;
        }

        if (!setupDisplay())
        {
            error("Could not setup display!");
            exit();
        }

        int status=DDSController.ERROR_OK;
        while (true)
        {
            ddsController.clearAll();
            ddsController.setCurrentRecord("MAIN");
            status=ddsController.write(-1); // No starting line, so pass a -1
            if (status!=DDSController.ERROR_OK)
            {
                error("Problem writing main menu record, reason='"+DDSController.getStatusDescription(status)+"'");
                exit();
            }
            status=ddsController.read();
            if (status!=DDSController.ERROR_OK)
            {
                error("Problem reading main menu record, reason='"+DDSController.getStatusDescription(status)+"'");
                exit();
            }

            String value=ddsController.getFieldValueAsString("SELECTION");
            if (value==null) continue;

            value=value.trim();
            if (value.length()==0) continue;

            char ch=value.charAt(0);
            switch (ch)
            {
                case '1': status=performRecordPositioningTests();
                          break;
                case '2': status=performDisplayAttributeTests();
                          break;
                case '3': status=performOptionIndicatorTests();
                          break;
                case '4': status=performMessageTests();
                          break;
                case '5': status=performCursorPositioningTests();
                          if (status!=DDSController.ERROR_OK) break;
                          status=performEntryFieldAttributeTests();
                          if (status!=DDSController.ERROR_OK) break;
                          status=performInputAttributeTests();
                          break;
                case '6': status=performKeyboardControlTests();
                          break;
                case '7': status=performDataValidationTests();
                          break;
                case '8': status=performEditKeywordTests();
                          break;
                case '9': status=performResponseIndicatorTests();
                          break;
                case 'A': status=performMiscOverlayTests();
                          break;
                case 'B': status=performDateTimeTests();
                          break;
                case 'C': status=performErrorMessageTests();
                          break;
                case 'D': status=performHelpRecordTests();
                          break;
                case 'E': status=performSubfileRecordTests();
                          break;
                case 'Q': exit();
                          break;
                default:
                          break;
            }

            if (DDSController.isError(status))
            {
                error("Abnormal exit, reason='"+DDSController.getStatusDescription(status)+"'");
                exit();
            }
        }
    }

    static public void main(String[] args)
    {
        DDSJava ddsJava=new DDSJava();
        ddsJava.performTests();
    }
}


