/*
 * CallETPFromJava
 * Example of invoking an Heirloom Computing Elastic 
 * Transaction Platform transaction from Java.
 * 
 * This does not have to be contained in an Elastic COBOL project,
 * a standard Java project will suffice with the Java Build Path:
 * including 3 additional JARs:
 * - openejb-client-4.7.4.jar (http://tomee.apache.org/downloads.html)
 * - javaee-api-7.0.jar (http://tomee.apache.org/downloads.html)
 * - ecobol.jar (part of Elastic COBOL)
 * 
 * EJB Context Properties (using OpenEJB factory objects):
 * - java.naming.factory.initial=org.openejb.client.RemoteInitialContextFactory
 * - java.naming.provider.url=ejbd://localhost:4201
 * - java.naming.security.principal=userid
 * - java.naming.security.credentials=password
 * 
 * By Heirloom convention, the lookup for the remote interface object for
 * ETP SYSID 'ABCD' (i.e., "CICS region" ABCD) is "ETPSYSL_etp2Remote"
 * 
 * Different property names will be required for JBOSS JNP,
 * Oracle WebLogics and IBM WebSphere client libraries.  security 
 * principal and credentials are required only if the JEE server is
 * running under control of a Security Manager and the client library
 * doesn't convey the local security context to the server.
 * 
 * Setup Required:  Start JEE server with an Elastic COBOL ETP
 * Deployment Project deployed to it as SYSID 'ETP2'  In this example,
 * the requested transaction ID is 'ELNK' and associated with 
 * the COBOL Program ID 'ETPELNK'.  It is a non-BMS transaction.
 * Upon completion, the program should execute the following: 
 *      EXEC CICS RETURN COMMAREA('something') END-EXEC
 *      
 */

import java.util.Properties;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import com.heirloomcomputing.etp.transaction.env.ejb.LinkDispatcherRemote;
import com.heirloomcomputing.etp.transaction.link.CommArea;

public class ETPCallFromJava {

    public static void main(String[] args) throws Exception {

    	final Properties props = new Properties();
    	String sysID = "etp2";	// the SYSID (i.e., CICS REGION) to connect
        String transID = null;;
        String programID = null;
        CommArea sendCommArea = null, returnCommArea = null;
        boolean syncOnReturn;
        String resultString = null;
        Object AnyThing = null;

    	props.setProperty(Context.INITIAL_CONTEXT_FACTORY, "org.openejb.client.RemoteInitialContextFactory");
    	props.setProperty(Context.PROVIDER_URL, "ejbd://localhost:4201");
    	//props.setProperty(Context.SECURITY_PRINCIPAL, "system");
    	//props.setProperty(Context.SECURITY_CREDENTIALS, "manager");

    	System.out.println("ETP call From Java example");
    	System.out.println();
    	
    	System.out.println("EJB Context Lookup properties:");
    	System.out.println(props.toString());
        final Context context = new InitialContext(props);

        // Get a new EJB Context to the ETP with SYSID "ETP2"
        try {
	        AnyThing = context.lookup("ETPSYSL_" + sysID + "Remote");
        } catch (NamingException e) {
        	System.err.println("ETP server not started or region name '" + sysID + "' not deployed and started.");
        	e.printStackTrace(System.err);
        	return;
        }
 
        // the object returned must be a com.heirloomcomputing.etp.transaction.env.ejb.LinkDispatcherRemote
        // or this cast generates an error.
        LinkDispatcherRemote ETP2System = (LinkDispatcherRemote) AnyThing;

        // invoke an ETP transaction from Java
        
        // call either a 4-char trans ID or a COBOL program ID (as a Java class: lower case; replace "-" with "_")
        //transID = "ELNK";
        programID = "etpetrn";
        sendCommArea = new CommArea("Hello ETP transaction from Java".getBytes()); // note:  Default CODE Page
        syncOnReturn = false; // require this program to commit() rather than the transaction itself.
        
        returnCommArea = ETP2System.link(transID, programID, sendCommArea, syncOnReturn);
        if (returnCommArea.getLength() ==  0) {
        	// we're expecting something from the transaction
        	ETP2System.rollback(); // or abort() or abend() transaction
        } else {
        	// the transaction returned something
        	if (transID != null) {
        		System.out.println("ETP2 transaction " + transID + " returned " + returnCommArea.getLength() + " bytes.");
        	} else {
        		System.out.println("ETP2 program " + programID + " returned " + returnCommArea.getLength() + " bytes.");
        	}

        	// resultString = new String(returnCommArea.toString()); // default code page conversion
        	resultString = new String(returnCommArea.toByteArray(), "UTF-8"); // specific code page conversion
        	System.out.println("    returned communications area:" + resultString);

        	// ETP2System.prepare();  // 2-phase prepare() transaction
        	ETP2System.commit(); // 1-phase or 2-phase commit() transaction
        }
    }
}
