      *
      * TCPIPSERVER
      *
      * Copyright (C) 2010-2023 Heirloom Computing Inc.  All Rights Reserved.
      *
      * This file and associated files are copyrighted information
      * of Heirloom Computing.  Permission is granted for usage in
      * conjunction with the Elastic COBOL product.
      *
      * This is an example program providing time of day services
      * along with a banner message to a client machine connecting
      * via either another Elastic COBOL program or TELNET.
      *
      * This example should be used for writing COBOL server programs.
      *
      * This demo uses the SCREEN SECTION to provide a visual status
      * of 14 socket connections (those which fit on a screen); more
      * connections are allowed, but will not be shown.
      *
      * It also includes a clock which is updated once per second.
      * It includes a date, which is updated only once per execution.
      * (Add the date display into the clock-display for constant
      *  date updates.)
      *
      * It also includes an advertising banner for Elastic COBOL near
      * the bottom of the screen.
      *
      
       IDENTIFICATION DIVISION.
       PROGRAM-ID.
           TCPIPSERVER.
      
       ENVIRONMENT DIVISION.
       CONFIGURATION SECTION.
      *
      * The REPOSITORY is a COBOL 2000 feature implemented in
      * Elastic COBOL.  It provides a listing of all classes
      * to be used by this program.  Elastic COBOL has direct access
      * using the COBOL 2000 features to Java, for Java objects
      * and Java Beans.
      
       REPOSITORY.
           CLASS JAVA-LANG-THREAD IS "java.lang.Thread"
      	   CLASS JAVA-NET-SOCKET IS "java.net.Socket"
           .
      
       INPUT-OUTPUT SECTION.
       FILE-CONTROL.
      
      * The file selection and usage is standard, but the device it
      * is assigned to exists only with Elastic COBOL.  The server:
      * device opens a TCP/IP server port at the given port number
      * and then waits for connections whenever opened; when
      * opened, the file automatically becomes a standard TCP/IP
      * socket to a client for that thread.  Open the server from 
      * another thread, get another client socket.
      
       SELECT TCP-FILE ASSIGN TO "server:3456"
           ORGANIZATION LINE SEQUENTIAL
           FILE STATUS IS TCP-FILE-STATUS
           .
      
       DATA DIVISION.
       FILE SECTION.
      *
      * The template for the record to be sent to the client.
      *
       FD  TCP-FILE.
       01  PRINT-REC.
           05 PRINT-REC-TN PIC XX.
           05 FILLER PIC X VALUE SPACES.
           05 PRINT-REC-TOD PIC X(22).
           05 FILLER PIC X VALUE SPACES.
           05 BANNER-F PIC X(56).
      
      * Elastic COBOL supports the X/Open Screen Section for
      * GUI's AND for dumb terminals.
      
       SCREEN SECTION.
       01 STATUS-SCREEN.
         05 STATUS-BLANK
            BACKGROUND-COLOR WHITE 
            FOREGROUND-COLOR BLACK
            BLANK SCREEN.
      	    10 VALUE "Elastic COBOL Time Of Day Server"
      	       LINE 1 COLUMN 27 BLINK HIGHLIGHT.
      	    10 VALUE "Elastic COBOL and Time Of Day Server " &
                    "Copyright (C) Heirloom Computing Inc"
                    LINE 2 COLUMN 5 HIGHLIGHT.
      	    10 VALUE "Thread"
      	       COLUMN 1 LINE 4 UNDERLINE.
      	    10 VALUE "Status"
      	       COLUMN 8 LINE 4 UNDERLINE.
      	    10 VALUE 
      	       "Socket                            "
      	       & "                               "
      	       COLUMN 15 LINE 4 UNDERLINE.
      	    10 CLOCK-DATE PIC X(9) COLUMN 60 LINE 22 HIGHLIGHT.
      	    10 CLOCK-TIME PIC X(8) COLUMN 70 LINE 22 HIGHLIGHT.
      	    10 VALUE "Port 3456" COLUMN 1 LINE 22 HIGHLIGHT UNDERLINE.
      
      	    10 BANNER PIC X(80) COLUMN 1 LINE 20 HIGHLIGHT VALUE SPACES.
      * 
      * LOCAL-STORAGE is an Elastic COBOL extension.
      *
      * Every instance of an Elastic COBOL program and every thread
      * of an Elastic COBOL program has its own local storage.
      *
      * In this example, the LOCAL STORAGE is used so that each
      * thread of the server does not interfere with another thread
      * of the server.
      *
       LOCAL-STORAGE SECTION.
       01 TCP-DATA.
         05 TCP-THREAD PIC 99.
         05 FILLER PIC X VALUE SPACES.
         05 TCP-DATA-INNER PIC X(22).
      	  05 FILLER PIC X VALUE SPACES.
      	  05 BANNER-F PIC X(52) VALUE 
           "Heirloom Computing Inc, http://heirloom.cc".
      
       01 CURRENT-SOCKET OBJECT REFERENCE JAVA-NET-SOCKET.
       01 ACTIVE-LINE PIC 99.
       01 SOCKET-LINE PIC X(65).
      
       WORKING-STORAGE SECTION.
       01 TCP-FILE-STATUS PIC 99.
      
       PROCEDURE DIVISION.
      * 
      * The first paragraph performs some setup operations, including
      * setting the initial date and time, and displaying the screen.
      *
       GREETING.
           MOVE CURRENT-DATE TO CLOCK-DATE
      	    MOVE CURRENT-TIME TO CLOCK-TIME
      	    DISPLAY STATUS-SCREEN
      * Start the clock update thread and continue onwards (#2)
           THREAD CLOCK-UPDATE
      * Start the banner update (#3)
           THREAD BANNER-UPDATE
      * Start the server (#4)
      	    THREAD OPEN-FILE
      * End this thread (#1)
      	    EXIT PROGRAM
      	    .
      
       OPEN-FILE.
      *
      * Wait for a connection; server: connections wait for a
      * client connection, and then are treated as normal
      * socket connections.  They may be reopened for another
      * client safely from another thread.
      *
      
           OPEN OUTPUT TCP-FILE
      
           IF TCP-FILE-STATUS >= 10 THEN
               DISPLAY "Could not open server socket: check "
                       "if other instance is running." UPON SYSOUT
               STOP ALL RUN
           END-IF
      
      *	   
      * Start a new thread of execution with OPEN-FILE and continue.
      *
           THREAD OPEN-FILE
      	    .
      *
      * The program falls through to here in one thread of execution.
      *
      
       SET-THREAD-NUMBER.
      
      * We want to know which thread we are.
           SET TCP-THREAD TO CURRENT-THREAD
      * Adjust for the extra threads we created
           SUBTRACT 2 FROM TCP-THREAD
      
      *
      * Obtain the Java Object for the TCP/IP Socket we have.
      * This is not necessary to use the Socket, but it allows
      * us to display a nice representation of the Socket
      * on the screen.  INVOKE'ing methods on the returned
      * object allows us to find out where the Socket is coming
      * from, etc.
      *
      
           SET CURRENT-SOCKET TO FUNCTION NATIVE(TCP-FILE)
      
      *
      * Set the display line
      *
           ADD 5 TO TCP-THREAD GIVING ACTIVE-LINE
      
      *
      * Only show the status until it fills the screen  
      *
      	    IF ACTIVE-LINE < 19 THEN
      *
      * Set an alpanumeric to an object to obtain a readable version
      * of its contents.  We could have displayed the object directly
      * but then it could have overflowed the right bounds of the
      * screen on occasion.  This way we know that the resulting
      * text will fit on the screen.
      *
      
               SET SOCKET-LINE TO CURRENT-SOCKET
      	        DISPLAY TCP-THREAD  AT COLUMN 1  LINE ACTIVE-LINE
      	        DISPLAY "Active"    AT COLUMN 8  LINE ACTIVE-LINE
      	                            HIGHLIGHT
      	        DISPLAY SOCKET-LINE AT COLUMN 15 LINE ACTIVE-LINE
      	    END-IF
      	    .
      
      * Write the data to the client.
      
       WRITE-TO-CLIENT.
      
      * Move the current date to local storage for the file.
      
           MOVE FUNCTION CURRENT-DATE TO TCP-DATA-INNER
      
      * For each client to have different data, the data must
      * be in local storage, and that means writing from
      * and reading into separate buffers.  The (RE)WRITE FROM
      * and READ INTO verbs are safely synchronized for the file.
      *
      * Do NOT use the FILE STATUS from multi-threaded programs
      * as FILE STATUS cannot currently be in LOCAL-STORAGE;
      * if the network connection is closed from the other
      * side, EOF will be set for READ's, and WRITE's can
      * check INVALID KEY for this.
      *
      * If a program gives NullPointerException's when multi-
      * threaded, there's a good chance the program didn't
      * use LOCAL-STORAGE where necessary.  If one thread is
      * manipulating a value while another is also trying
      * to manipulate it, data can be lost; this will not
      * occur with LOCAL-STORAGE.
      
           WRITE PRINT-REC FROM TCP-DATA AFTER ADVANCING 1 LINE
      	    INVALID KEY
      *
      * Only show the status until it fills the screen  
      *
               IF ACTIVE-LINE < 19 THEN
      	            SET SOCKET-LINE TO CURRENT-SOCKET
      	            DISPLAY "  "        AT COLUMN 1 LINE ACTIVE-LINE
      	            DISPLAY "------"    AT COLUMN 8 LINE ACTIVE-LINE
      	            DISPLAY SOCKET-LINE AT COLUMN 15 LINE ACTIVE-LINE
               END-IF
      *
      *     This close affects only the current client.  To have the
      *     close affect the server socket itself, CLOSE WITH LOCK.
      *
      	        CLOSE TCP-FILE
      *
      *     Actually, this exits only the current thread.  This program
      *     must be stopped by a CTRL-C or kill command from the OS
      *     as it is a server meant to run continuously.  (If running
      *     this on a GUI, closing the display console also works.)
      *
      	        EXIT PROGRAM
      	    END-WRITE
      *
      * Update the client once per second
      *
      	    INVOKE JAVA-LANG-THREAD "sleep" USING BY VALUE 1000
      
      	    GO TO WRITE-TO-CLIENT
      	    .
      
       DONE-WRITING.
      *
      * Close port to end connection from this side.
      *
      	    CLOSE TCP-FILE. 
      *
      * Exit this thread.
      *
      	    EXIT PROGRAM.
      
      *
      * Update the visible clock once per second.
      *	   
       CLOCK-UPDATE.
           MOVE CURRENT-TIME TO CLOCK-TIME
      	    DISPLAY CLOCK-TIME
      	    INVOKE JAVA-LANG-THREAD "sleep" USING BY VALUE 1000
      	    GO TO CLOCK-UPDATE
      	    .
      
      *
      * An advertising banner for Elastic COBOL across the bottom of
      * the screen.  It displays a message and then waits a
      * number of seconds before displaying another message.
      * It repeats after showing all banner messages.
      * 
       BANNER-UPDATE.
           MOVE "Heirloom Computing's Elastic COBOL " TO BANNER
      	   DISPLAY BANNER
      	   INVOKE JAVA-LANG-THREAD "sleep" USING BY VALUE 8000
      
           MOVE "Elastic COBOL and Micro Focus file support." TO BANNER
           DISPLAY BANNER
           INVOKE JAVA-LANG-THREAD "sleep" USING BY VALUE 8000
      
           MOVE "X/Open SCREEN SECTION for Terminals and GUI's"
      	         TO BANNER
           DISPLAY BANNER
           INVOKE JAVA-LANG-THREAD "sleep" USING BY VALUE 4000
      
           MOVE "Multithreading using THREAD, PERFORM THREAD,"
                    & " CALL THREAD statement" TO BANNER
           DISPLAY BANNER
           INVOKE JAVA-LANG-THREAD "sleep" USING BY VALUE 4000
      
           MOVE "TCP/IP control using file statements, " 
           & "BOTH client and server!" TO BANNER
           DISPLAY BANNER
           INVOKE JAVA-LANG-THREAD "sleep" USING BY VALUE 4000
      
           MOVE "This demonstration in COBOL using SCREEN SECTION, "
      	         & "THREAD, and SERVER: files!" TO BANNER
           DISPLAY BANNER
           INVOKE JAVA-LANG-THREAD "sleep" USING BY VALUE 4000
      
           MOVE "Common IBM, HP, AcuCOBOL, X/Open, Micro Focus " 
                 & "extensions!  "
      	         TO BANNER
           DISPLAY BANNER
           INVOKE JAVA-LANG-THREAD "sleep" USING BY VALUE 4000
      
           MOVE "INVOKE and OBJECT REFERENCES to access Java API's!"
       	         TO BANNER
      	   DISPLAY BANNER
      	   INVOKE JAVA-LANG-THREAD "sleep" USING BY VALUE 4000
      
           GO TO BANNER-UPDATE
      	   .
